<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Commission;
use App\Models\Mining;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class PurchaseController extends Controller
{
    /**
     * প্যাকেজ কনফার্মেশন + ক্রয় প্রসেস
     */
    public function purchaseConfirmation($id)
    {
        $package = Package::find($id);
        $user    = Auth::user();

        // ✅ প্যাকেজ আছে কি না
        if (!$package) {
            return back()->with('error', "VIP Package not found.");
        }

        // ✅ প্যাকেজ অ্যাক্টিভ না থাকলে
        if ($package->status != 'active') {
            return back()->with('error', $package->name . " is inactive.");
        }

        // ==============================
        // ✅ LIMIT PER USER (max_per_user)
        // ==============================
        // 0 = Unlimited; >0 হলে ওই সংখ্যায় সীমাবদ্ধ
        $maxPerUser = (int) ($package->max_per_user ?? 0);

        // ইউজার এই প্যাকেজটা কতবার কিনেছে (প্রয়োজনে status ফিল্টার সমন্বয় করতে পারো)
        $alreadyBought = Purchase::where('user_id', $user->id)
                            ->where('package_id', $package->id)
                            ->count();

        if ($maxPerUser > 0 && $alreadyBought >= $maxPerUser) {
            return back()->with('error', "You can buy this VIP a maximum of {$maxPerUser} times.");
        }

        // ==============================
        // ✅ REFERRAL (২য় বার থেকে রেফার লাগবে)
        // ==============================
        // আগের লজিকটা এমন ছিল যে প্রথমবারেও রেফার চেক হচ্ছিল।
        // এখনকার রুল: প্রথমবার ফ্রি, ২য় থেকে রেফার লাগবে।
        if ($package->referral_enabled) {
            // শুধুমাত্র যদি আগে অন্তত ১বার কেনা হয়ে থাকে, তাহলে রেফার চেক লাগবে
            if ($alreadyBought >= 1) {
                // তোমার সিস্টেম অনুযায়ী ডাইরেক্ট রেফার কাউন্ট
                $referralCount = User::where('ref_by', $user->ref_id)->count();
                $need          = (int) ($package->referral_required ?? 0);

                if ($need > 0 && $referralCount < $need) {
                    return back()->with(
                        'error',
                        "To purchase this VIP again, refer at least {$need} user(s). You currently have {$referralCount} referral(s)."
                    );
                }
            }
        }

        // ✅ ব্যালান্স চেক
        if ($package->price > $user->balance) {
            return back()->with('error', "Your balance is insufficient to purchase this package.");
        }

        // ✅ ব্যালান্স কমানো
        User::where('id', $user->id)->update([
            'balance' => $user->balance - $package->price,
        ]);

        // ✅ Purchase Table Create
        $purchase = new Purchase();
        $purchase->user_id          = $user->id;
        $purchase->package_id       = $package->id;
        $purchase->amount           = $package->price;
        $purchase->daily_income     = $package->commission_with_avg_amount / max(1, $package->validity);
        $purchase->date             = now()->format('d-m-Y H:i');
        $purchase->validity         = Carbon::now()->addDays($package->validity);
        $purchase->next_return_date = Carbon::now()->addHours(24);
        $purchase->total_return     = $package->validity;
        $purchase->running_status   = 'running';
        $purchase->status           = 'active';
        $purchase->save();

        // ✅ Ledger তৈরি
        $ledger = new UserLedger();
        $ledger->user_id       = $user->id;
        $ledger->reason        = 'vip_purchase';
        $ledger->perticulation = 'Your VIP purchase was successful. Thank you!';
        $ledger->amount        = $package->price;
        $ledger->credit        = $package->price; // (তোমার সিস্টেমে যেভাবে ছিল, 그대로 রাখলাম)
        $ledger->status        = 'approved';
        $ledger->date          = now()->format('Y-m-d H:i:s');
        $ledger->save();

        // ✅ রেফারেল কমিশন (আগের মতোই)
        $first_ref = User::where('ref_id', $user->ref_by)->first();
        if ($first_ref) {
            $amount = $package->price * setting('first_refer_commission') / 100;
            User::where('id', $first_ref->id)->update([
                'income' => $first_ref->income + $amount
            ]);

            $ledger = new UserLedger();
            $ledger->user_id                = $first_ref->id;
            $ledger->get_balance_from_user_id = $user->id;
            $ledger->reason                 = 'refer_bonus';
            $ledger->perticulation          = 'Congratulations! You received referral income.';
            $ledger->amount                 = $amount;
            $ledger->debit                  = $amount;
            $ledger->status                 = 'approved';
            $ledger->step                   = 'first';
            $ledger->date                   = now()->format('d-m-Y H:i');
            $ledger->save();

            $second_ref = User::where('ref_id', $first_ref->ref_by)->first();
            if ($second_ref) {
                $amount = $package->price * setting('second_refer_commission') / 100;
                User::where('id', $second_ref->id)->update([
                    'income' => $second_ref->income + $amount
                ]);

                $ledger = new UserLedger();
                $ledger->user_id                 = $second_ref->id;
                $ledger->get_balance_from_user_id = $user->id;
                $ledger->reason                  = 'refer_bonus';
                $ledger->perticulation           = 'Congratulations! You received second-level referral income.';
                $ledger->amount                  = $amount;
                $ledger->debit                   = $amount;
                $ledger->status                  = 'approved';
                $ledger->step                    = 'second';
                $ledger->date                    = now()->format('d-m-Y H:i');
                $ledger->save();

                $third_ref = User::where('ref_id', $second_ref->ref_by)->first();
                if ($third_ref) {
                    $amount = $package->price * setting('third_refer_commission') / 100;
                    User::where('id', $third_ref->id)->update([
                        'income' => $third_ref->income + $amount
                    ]);

                    $ledger = new UserLedger();
                    $ledger->user_id                 = $third_ref->id;
                    $ledger->get_balance_from_user_id = $user->id;
                    $ledger->reason                  = 'refer_bonus';
                    $ledger->perticulation           = 'Congratulations! You received third-level referral income.';
                    $ledger->amount                  = $amount;
                    $ledger->debit                   = $amount;
                    $ledger->status                  = 'approved';
                    $ledger->step                    = 'third';
                    $ledger->date                    = now()->format('d-m-Y H:i');
                    $ledger->save();
                }
            }
        }

        return redirect()->route('dashboard')->with('success', "Congratulations! Purchase successful.");
    }

    /**
     * ভিউ: VIP কনফার্ম
     */
    public function vip_confirm($vip_id)
    {
        $vip = Package::find($vip_id);
        return view('app.main.vip_confirm', compact('vip'));
    }

    /**
     * (তোমার আগের হেল্পার—নেইম রেখে দিলাম)
     */
    protected function ref_user($ref_by)
    {
        return User::where('ref_id', $ref_by)->first();
    }
}
