<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Package;
use App\Models\Purchase;
use App\Models\User;
use App\Models\UserLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class AdminController extends Controller
{
    // GET: /admin/login
    public function login()
    {
        if (Auth::guard('admin')->check()) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.auth.login');
    }

    public function salaryView()
    {
        return view('admin.salary');
    }

    /* --------------------  ✅ ADDED: fallback for old route  -------------------- */
    // কিছু রাউট বা ফর্ম যদি ভুলবশত AdminController::salary() কল করে, তাহলে ভিউ-ই দেখাবে
    public function salary()
    {
        return $this->salaryView();
    }

    /* --------------------  ✅ ADDED: handle POST submit  -------------------- */
    // POST: /admin/secured/salary-submit  (বা যেকোনো mapped POST route)
    public function salarySubmit(Request $request)
    {
        // TODO: আপনার প্রকৃত সেভ/ভ্যালিডেশন লজিক এখানে বসান
        // উদাহরণ:
        // $request->validate([...]);
        // Model::create([...]);

        return back()->with('success', 'Salary form submitted successfully.');
    }
    /* ------------------------------------------------------------------------- */

    // POST: /admin/login
    public function login_submit(Request $request)
    {
        $credentials = $request->only('email', 'password');

        if (Auth::guard('admin')->attempt($credentials)) {
            $admin = Auth::guard('admin')->user();

            if ($admin->type === 'admin') {
                return redirect()
                    ->route('admin.dashboard')
                    ->with('success', 'Logged In Successfully.');
            } else {
                Auth::guard('admin')->logout();
                return error_redirect(
                    'admin.login',
                    'error',
                    "Admin Credentials Very Secured Please Don't try Again."
                );
            }
        }

        return error_redirect('admin.login', 'error', 'Invalid credentials.');
    }

    // GET: /admin/logout
    public function logout()
    {
        if (Auth::guard('admin')->check()) {
            Auth::guard('admin')->logout();
            return redirect()
                ->route('admin.login')
                ->with('success', 'Logged out successfully.');
        }

        return error_redirect('admin.login', 'error', 'You are already logged out.');
    }

    // GET: /admin/dashboard
    public function dashboard()
    {
        return view('admin.dashboard');
    }

    // GET: /admin/profile
    public function profile()
    {
        return view('admin.profile.index');
    }

    // GET: /admin/profile/update
    public function profile_update()
    {
        $admin = Admin::first();
        return view('admin.profile.update-details', compact('admin'));
    }

    // POST: /admin/profile/update
    public function profile_update_submit(Request $request)
    {
        $admin = Admin::find(1);

        // আপনার uploadImage() হেল্পার আগের মতোই থাকছে
        $path = uploadImage(
            false,
            $request,
            'photo',
            'admin/assets/images/profile/',
            $admin->photo
        );

        $admin->photo   = $path ?? $admin->photo;
        $admin->name    = $request->name;
        $admin->email   = $request->email;
        $admin->phone   = $request->phone;
        $admin->address = $request->address;
        $admin->update();

        return redirect()
            ->route('admin.profile.update')
            ->with('success', 'Admin profile updated.');
    }

    // GET: /admin/change-password
    public function change_password()
    {
        $admin = admin()->user();
        return view('admin.profile.change-password', compact('admin'));
    }

    // POST: /admin/check-password (AJAX)
    public function check_password(Request $request)
    {
        $admin    = admin()->user();
        $password = $request->password;

        if (Hash::check($password, $admin->password)) {
            return response()->json([
                'message' => 'Password matched.',
                'status'  => true,
            ]);
        }

        return response()->json([
            'message' => 'Password dose not match.',
            'status'  => false,
        ]);
    }

    // POST: /admin/change-password
    public function change_password_submit(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'old_password'      => 'required',
            'new_password'      => 'required',
            'confirm_password'  => 'required',
        ]);

        if ($validate->fails()) {
            session()->put('errors', true);
            return redirect()
                ->route('admin.changepassword')
                ->withErrors($validate->errors());
        }

        $admin    = admin()->user();
        $password = $request->old_password;

        if (!Hash::check($password, $admin->password)) {
            return error_redirect(
                'admin.changepassword',
                'error',
                'Password dose not match'
            );
        }

        if (strlen($request->new_password) <= 5 || strlen($request->confirm_password) <= 5) {
            return error_redirect(
                'admin.changepassword',
                'error',
                'Password must be greater then 6 or equal.'
            );
        }

        if ($request->new_password !== $request->confirm_password) {
            return error_redirect(
                'admin.changepassword',
                'error',
                'New password and confirm password dose not match'
            );
        }

        $admin->password = Hash::make($request->new_password);
        $admin->update();

        return redirect()
            ->route('admin.changepassword')
            ->with('success', 'Password changed successfully');
    }
}
