<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Purchase;
use App\Models\User;

class CreditDailyIncome extends Command
{
    protected $signature = 'income:credit-daily';
    protected $description = 'Active purchases এর daily income ইউজারের একাউন্টে ক্রেডিট করে';

    public function handle()
    {
        $now = Carbon::now();

        $purchases = Purchase::where('status', 'active')
            ->where('running_status', 'running')
            ->where('next_return_date', '<=', $now)
            ->get();

        if ($purchases->isEmpty()) {
            $this->info('No due incomes to credit today.');
            return Command::SUCCESS;
        }

        foreach ($purchases as $purchase) {
            $user = User::find($purchase->user_id);
            if (!$user) continue;

            $dailyIncome = (float) $purchase->daily_income;
            $countReturn = (int) $purchase->count_return;
            $totalReturn = (int) $purchase->total_return;

            DB::transaction(function () use ($purchase, $user, $dailyIncome, $countReturn, $totalReturn) {
                // 1️⃣ ইউজারের ব্যালেন্সে ইনকাম যোগ
                $user->increment('balance', $dailyIncome);

                // 2️⃣ লগ (Transaction টেবিল থাকলে)
                if (class_exists(\App\Models\Transaction::class)) {
                    \App\Models\Transaction::create([
                        'user_id' => $user->id,
                        'type' => 'credit',
                        'amount' => $dailyIncome,
                        'remark' => 'Daily ROI',
                        'created_at' => now(),
                    ]);
                }

                // 3️⃣ পারচেজ আপডেট
                $purchase->count_return = $countReturn + 1;
                $purchase->next_return_date = Carbon::parse($purchase->next_return_date)->addDay();

                // মেয়াদ শেষ হলে বন্ধ করে দাও
                if ($purchase->count_return >= $totalReturn) {
                    $purchase->status = 'inactive';
                    $purchase->running_status = 'stop';
                }

                $purchase->save();
            });

            Log::info("Credited daily income for purchase #{$purchase->id}, user {$user->id}");
        }

        $this->info("✅ Daily income credited successfully for {$purchases->count()} purchases.");
        return Command::SUCCESS;
    }
}
